#!/usr/bin/env python3
"""
bf_pin_openssl.py
Bruteforce 6-digit PIN used as openssl enc -aes-256-cbc -a -salt -pbkdf2 -iter 10 -pass file:/tmp/secret

Usage:
  python3 bf_pin_openssl.py encrypted.b64  # encrypted.b64 contains base64 output from openssl
Options:
  -p NPROCS   number of processes to use (default: cpu_count)
  -r START:END  range of pins to try (inclusive, zero-padded), e.g. 000000:099999
"""

import sys, os, base64, hashlib, itertools
from multiprocessing import Pool, cpu_count
from Cryptodome.Cipher import AES

ITER = 10
DKLEN = 48  # 32 bytes key + 16 bytes IV
GZIP_MAGIC = b'\x1f\x8b\x08'

def extract_salt_and_ciphertext(b64data: bytes):
    raw = base64.b64decode(b64data)
    if raw.startswith(b"Salted__"):
        salt = raw[8:16]
        ciphertext = raw[16:]
        return salt, ciphertext
    else:
        raise ValueError("Input does not use OpenSSL 'Salted__' format")

def derive_key_iv(pin_bytes: bytes, salt: bytes, iterations: int = ITER, sha='sha256'):
    # returns (key, iv)
    keyiv = hashlib.pbkdf2_hmac(sha, pin_bytes, salt, iterations, dklen=DKLEN)
    return keyiv[:32], keyiv[32:48]

def try_pin(args):
    pin, salt, ciphertext = args
    pin_bytes = pin.encode()
    # try sha256 (recommended)
    for sha in ('sha256','sha1'):
        key, iv = derive_key_iv(pin_bytes, salt, ITER, sha=sha)
        try:
            cipher = AES.new(key, AES.MODE_CBC, iv)
            plain = cipher.decrypt(ciphertext)
            # quick check for gzip magic at start
            if plain.startswith(GZIP_MAGIC):
                return (pin, sha, plain)
        except Exception:
            pass
    return None

def chunks(iterable, n):
    it = iter(iterable)
    while True:
        chunk = list(itertools.islice(it, n))
        if not chunk:
            break
        yield chunk

def main():
    import argparse
    p = argparse.ArgumentParser()
    p.add_argument('infile', help='file containing base64 OpenSSL output')
    p.add_argument('-p', '--procs', type=int, default=cpu_count(), help='number of processes')
    p.add_argument('-r', '--range', default='000000:999999', help='pin range START:END (inclusive), zero-padded')
    args = p.parse_args()

    with open(args.infile, 'rb') as f:
        b64data = f.read().strip()

    try:
        salt, ciphertext = extract_salt_and_ciphertext(b64data)
    except ValueError as e:
        print("Error:", e)
        sys.exit(2)

    start_s, end_s = args.range.split(':')
    start = int(start_s)
    end = int(end_s)
    if start < 0 or end > 999999 or start > end:
        print("Invalid range")
        sys.exit(2)

    pins = (f"{i:06d}" for i in range(start, end+1))

    print(f"Salt (hex): {salt.hex()}")
    print(f"Trying pins {start_s} → {end_s} with {args.procs} processes... (PBKDF2 iter={ITER})")

    pool = Pool(processes=args.procs)
    try:
        # map in chunks to reduce IPC overhead - create args tuples
        def gen_args():
            for pin in pins:
                yield (pin, salt, ciphertext)

        for res in pool.imap_unordered(try_pin, gen_args(), chunksize=256):
            if res:
                pin, sha, plain = res
                print(f"\nFOUND PIN: {pin} (used {sha})")
                outname = f"decrypted_{pin}.tgz"
                with open(outname, 'wb') as out:
                    out.write(plain)
                print(f"Decrypted file written to: {outname}")
                pool.terminate()
                return
        print("PIN not found in given range.")
    finally:
        pool.close()
        pool.join()

if __name__ == '__main__':
    main()
