#!/usr/bin/env python3
"""
bf_pin_openssl_text.py
Bruteforce 6-digit PIN used as openssl enc -aes-256-cbc -a -salt -pbkdf2 -iter 10 -pass file:/tmp/secret
This variant expects the decrypted payload to be plain ASCII/UTF-8 text (not gzip).

Usage:
  python3 bf_pin_openssl_text.py encrypted.b64
Options:
  -p NPROCS   number of processes to use (default: cpu_count)
  -r START:END  range of pins to try (inclusive, zero-padded), e.g. 000000:099999
"""

import sys, os, base64, hashlib, itertools, string
from multiprocessing import Pool, cpu_count
from Cryptodome.Cipher import AES

ITER = 10
DKLEN = 48  # 32 bytes key + 16 bytes IV

def extract_salt_and_ciphertext(b64data: bytes):
    raw = base64.b64decode(b64data)
    if raw.startswith(b"Salted__"):
        salt = raw[8:16]
        ciphertext = raw[16:]
        return salt, ciphertext
    else:
        raise ValueError("Input does not use OpenSSL 'Salted__' format")

def derive_key_iv(pin_bytes: bytes, salt: bytes, iterations: int = ITER, sha='sha256'):
    # returns (key, iv)
    keyiv = hashlib.pbkdf2_hmac(sha, pin_bytes, salt, iterations, dklen=DKLEN)
    return keyiv[:32], keyiv[32:48]

def pkcs7_unpad(data: bytes):
    if not data:
        raise ValueError("Empty plaintext (no padding).")
    pad = data[-1]
    if pad < 1 or pad > AES.block_size:
        raise ValueError("Invalid padding length.")
    if data[-pad:] != bytes([pad]) * pad:
        raise ValueError("Invalid PKCS#7 padding bytes.")
    return data[:-pad]

PRINTABLE = set(bytes(string.printable, 'ascii'))

def is_likely_text(b: bytes, min_print_ratio=0.85):
    # try decode utf-8 (most likely). If fails, return False.
    try:
        s = b.decode('utf-8')
    except Exception:
        return False
    if not s:
        return False
    # count printable characters (including whitespace)
    total = len(s)
    printable = sum(1 for ch in s if ch in string.printable)
    ratio = printable / total
    # also require presence of at least one whitespace/newline and some letters/digits
    has_space = any(c.isspace() for c in s)
    has_alnum = any(c.isalnum() for c in s)
    return ratio >= min_print_ratio and has_space and has_alnum

def try_pin(args):
    pin, salt, ciphertext = args
    pin_bytes = pin.encode()
    for sha in ('sha256', 'sha1'):
        key, iv = derive_key_iv(pin_bytes, salt, ITER, sha=sha)
        try:
            cipher = AES.new(key, AES.MODE_CBC, iv)
            plain = cipher.decrypt(ciphertext)
            # try to unpad (PKCS#7) — if invalid padding, it's probably wrong key
            try:
                plain_u = pkcs7_unpad(plain)
            except Exception:
                continue
            # quick check for likely text
            if is_likely_text(plain_u):
                # return raw bytes and decoded text for convenience
                return (pin, sha, plain_u, plain_u.decode('utf-8'))
        except Exception:
            pass
    return None

def chunks(iterable, n):
    it = iter(iterable)
    while True:
        chunk = list(itertools.islice(it, n))
        if not chunk:
            break
        yield chunk

def main():
    import argparse
    p = argparse.ArgumentParser()
    p.add_argument('infile', help='file containing base64 OpenSSL output')
    p.add_argument('-p', '--procs', type=int, default=cpu_count(), help='number of processes')
    p.add_argument('-r', '--range', default='000000:999999', help='pin range START:END (inclusive), zero-padded')
    args = p.parse_args()

    with open(args.infile, 'rb') as f:
        b64data = f.read().strip()

    try:
        salt, ciphertext = extract_salt_and_ciphertext(b64data)
    except ValueError as e:
        print("Error:", e)
        sys.exit(2)

    start_s, end_s = args.range.split(':')
    start = int(start_s)
    end = int(end_s)
    if start < 0 or end > 999999 or start > end:
        print("Invalid range")
        sys.exit(2)

    pins = (f"{i:06d}" for i in range(start, end+1))

    print(f"Salt (hex): {salt.hex()}")
    print(f"Trying pins {start_s} → {end_s} with {args.procs} processes... (PBKDF2 iter={ITER})")

    pool = Pool(processes=args.procs)
    try:
        def gen_args():
            for pin in pins:
                yield (pin, salt, ciphertext)

        for res in pool.imap_unordered(try_pin, gen_args(), chunksize=256):
            if res:
                pin, sha, plain_bytes, plain_text = res
                print(f"\nFOUND PIN: {pin} (used {sha})")
                outname = f"decrypted_{pin}.txt"
                with open(outname, 'w', encoding='utf-8') as out:
                    out.write(plain_text)
                print(f"Decrypted text written to: {outname}")
                # also print a short preview to stdout
                preview = plain_text[:1000]
                print("\nPreview (first 1000 chars):\n")
                print(preview)
                pool.terminate()
                return
        print("PIN not found in given range.")
    finally:
        pool.close()
        pool.join()

if __name__ == '__main__':
    main()
