#!/usr/bin/env python3
import zlib
import itertools
import string

# RC4 implementation
def rc4(key: bytes, data: bytes) -> bytes:
    S = list(range(256))
    j = 0
    for i in range(256):
        j = (j + S[i] + key[i % len(key)]) & 0xFF
        S[i], S[j] = S[j], S[i]
    i = j = 0
    out = bytearray()
    for b in data:
        i = (i + 1) & 0xFF
        j = (j + S[i]) & 0xFF
        S[i], S[j] = S[j], S[i]
        K = S[(S[i] + S[j]) & 0xFF]
        out.append(b ^ K)
    return bytes(out)

# Helpers
def crc32_b(data: bytes):
    return zlib.crc32(data) & 0xFFFFFFFF

def printable_ascii(bs):
    return ''.join(chr(b) if 32 <= b < 127 else f"\\x{b:02x}" for b in bs)

# Inputs (from toi)
msg_hex = "5a15339de0ba7121cb056a8aca36b2990afb239a17c9572996"
msg = bytes.fromhex(msg_hex)  # 25 bytes
msg24 = msg[1:]               # 24 bytes (without leading 'Z')
expected = "a6dbacc5"
expected_int = int(expected, 16)

# Key candidates to try quickly
key_candidates = [
    b"ThreeLittleBirds",
    b"ThreeLittleBirds\x00",
    b"ThreeLittleBirds ",
    b"threelittlebirds",
    b"ThreeLittleBirds\n",
    b"ThreeLittleBirds\r",
]

# Variants to test: RC4 -> CRC32, CRC32 -> RC4, CRC32(message), RC4(message) etc.
def run_variants():
    print("== Deterministic key candidates ==")
    for k in key_candidates:
        # rc4(msg) then crc32
        r = rc4(k, msg)
        c = crc32_b(r)
        print(f"key={k!r} len={len(k)}  rc4(msg).hex()={r.hex()}")
        print(f"  crc32(rc4(msg)) = {c:08x}  match? {c == expected_int}")
        # rc4(msg24)
        r2 = rc4(k, msg24)
        c2 = crc32_b(r2)
        print(f"  crc32(rc4(msg24)) = {c2:08x}  match? {c2 == expected_int}")
        # crc of msg directly
        c3 = crc32_b(msg)
        print(f"  crc32(msg) = {c3:08x}")
        print("-"*60)

    print("\n== Simple transform attempts (xor all bytes by 0..255, invert order) ==")
    for k in key_candidates:
        r = rc4(k, msg)
        # try CRC on inverted bytes
        c_inv = crc32_b(r[::-1])
        print(f"key={k!r} crc32(rc4(msg)[::-1]) = {c_inv:08x}  match? {c_inv == expected_int}")
        # try single-byte XOR of rc4 output by a few values
        for xorval in (0,1,0x20,0x80,0xff):
            rx = bytes(b ^ xorval for b in r)
            cx = crc32_b(rx)
            if cx == expected_int:
                print(f"FOUND: key={k!r} xorval={xorval:02x} -> crc32 match; rc4_hex={r.hex()} xor_hex={rx.hex()}")
        print("-"*30)

    print("\n== CRC32 of RC4 keystream only (keystream = rc4(key, msg) XOR msg) ==")
    for k in key_candidates:
        out = rc4(k, msg)
        keystream = bytes(a ^ b for a,b in zip(out, msg))
        c = crc32_b(keystream)
        print(f"{k!r} crc32(keystream)={c:08x}")
    print("-"*60)

# Small bruteforce: try short ascii keys up to length N over a charset (careful: combinatorial!)
def brute_force_short_keys(max_len=5, charset=string.ascii_lowercase+string.digits):
    target = expected_int
    tried = 0
    print(f"Bruteforcing keys up to length {max_len} (charset len={len(charset)}) ...")
    for L in range(1, max_len+1):
        for key_tuple in itertools.product(charset, repeat=L):
            k = ''.join(key_tuple).encode()
            tried += 1
            r = rc4(k, msg)
            if crc32_b(r) == target:
                print(">> FOUND key:", k)
                return k
            if tried % 200000 == 0:
                print(f"  tried {tried} keys (current key length {L})")
    print("Bruteforce finished, no key found in this space.")
    return None

if __name__ == "__main__":
    run_variants()
    # brute_force_short_keys(max_len=4, charset="abc123")  # uncomment if you want to test small spaces
